#!/bin/bash

# This script generates benchmarks for performance testing of openllet temporal-query.
# It acts as a wrapper around tobm, creating benchmarks for an increasing scaling parameter.

VERBOSE=0                     # If > 0, prints verbose output of tobm
INFO=1                        # If > 0, prints info output of this script
SCENARIO="all"                # one of: "t-crossing", "x-crossing", "all"
N_MAX=5                       # Maximum scaling parameter
N_STEP_SIZE=1                 # Step size of the scaling parameter
HERTZ=10                      # Sampling rate of each benchmark scenario, in 1/s
DURATION=20                   # Duration of each benchmark scenario, in s
SEED=0                        # Seed to be used in random number generation of simulation
GEOMETRIES=1                  # Whether to create a version with geometries included alongside the original benchmarks.
AUGMENT=1                     # Whether to augment the resulting scenarios by semantic information.
CLEAN=0                       # Whether to clean up / remove all already existing benchmarks in the output directory.
DIRECTORY="benchmarks/aboxes" # Directory to write the output benchmarks to.

function initialize() {
	while getopts hvcgiat:d:f:n:s:o: flag
	do
		case "${flag}" in
			h) usage;;
			v) VERBOSE=1;;
			c) CLEAN=1;;
			g) GEOMETRIES=1;;
			a) AUGMENT=0;;
			i) INFO=0;;
			t) SCENARIO=${OPTARG};;
			d) DURATION=${OPTARG};;
			f) HERTZ=${OPTARG};;
			n) N_MAX=${OPTARG};;
			s) STEP_SIZE=${OPTARG};;
			o) DIRECTORY=${OPTARG};;
	    esac
	done
}

function usage() {
	cat <<HELP_USAGE
$0 [-hvciag] [-t <t-crossing|x-crossing|all>] [-d <float>] [-f <int>] [-n <int>] [-s <int>] [-o <string>]

This script generates benchmarks for performance testing of openllet temporal-query.
If run without any input, it creates the default set of benchmarks into the benchmarks directory.
Within this directory, benchmarks are stored in a folder structure of the format T/S/N where
T is the benchmark type, S the employed seed, and N the employed scaling factor.
Note: By default, this script will *not* overwrite already existing benchmarks.

Options:

-h      Shows this help message.
-v      Enables verbose output.
-c      Removes (cleans up) already existing benchmarks in the output directory.
-i      Disables information output of this script.
-a      Disables semantic augmentation of the benchmark scenarios
-g      Enables the creation of a version with geometries included alongside the original benchmark,
        in a folder called T/S/N_geom. Note: this will double the run-time.
-t      A specific type of benchmark (one of: "t-crossing", "x-crossing", "all", default: $SCENARIO)
-d      Exact duration of each benchmark scenario in seconds (default: $DURATION)
-f      Sampling frequency of each benchmark scenario in Hertz (default: $HERTZ)
-n      Maximum benchmark scenario size (default: $N_MAX)
-s      Step size for iterating over the benchmark scenario size (default: $N_STEP_SIZE)
-o      Output directory for the benchmarks (default: $DIRECTORY)
HELP_USAGE
    exit 0
}

function generate_benchmark_for_type() {
	benchmark_type=$1
	if [ $INFO -gt 0 ]
	then
		echo "$0 - $(date) - INFO: Generating benchmarks for $benchmark_type"
	fi
	if [[ "$benchmark_type" != "x-crossing" && "$benchmark_type" != "t-crossing" ]]
	then
		echo "$0 - $(date) - FATAL: Got invalid benchmark type $benchmark_type"
		exit 1
	fi
	directory_check $benchmark_type
	for n in $(seq 1 $N_STEP_SIZE $N_MAX)
	do
		dir="$DIRECTORY/$benchmark_type/$SEED/$n"
		if [ $INFO -gt 0 ]
		then
			echo "$0 - $(date) - INFO: Creating benchmark $n/$N_MAX into $dir"
		fi
		if [ $AUGMENT -gt 0 ]
		then
			augment_flag="-e"
		fi
		if [ $VERBOSE -gt 0 ]
		then
			verbose_flag="-v"
		fi
		if [ $GEOMETRIES -gt 0 ]
		then
			geometry_flag="-g"
			dir="${dir}_geom"
		fi
		if [ $INFO -gt 0 ]
		then
			echo "$0 - $(date) - INFO: Starting tobm"
		fi
		tobm $INFO_flag $augment_flag $geometry_flag -s $SEED -d $DURATION -hz $HERTZ -t "$benchmark_type" -n $n "$dir"
		if [ $? -ne 0 ]
		then
			echo "$0 - $(date) - FATAL: Benchmark $n could not be generated"
			exit 1
		fi
		if [ $INFO -gt 0 ]
		then
			echo "$0 - $(date) - INFO: Finished benchmark $n"
		fi
	done
	if [ $INFO -gt 0 ]
	then
		echo "$0 - $(date) - INFO: Finished benchmark generation for $benchmark_type"
	fi
}

function generate_benchmarks() {
	if [ "$SCENARIO" == "all" ]
	then
		generate_benchmark_for_type "x-crossing"
		if [ $? -eq 0 ]
		then
			generate_benchmark_for_type "t-crossing"
		fi
	elif [[ "$SCENARIO" == "x-crossing" || "$SCENARIO" == "t-crossing" ]]
	then
		generate_benchmark_for_type "$SCENARIO"
	else
		echo "$0 - $(date) - FATAL: Got invalid benchmark type $SCENARIO"
		exit 1
	fi
}

function directory_check() {
	non_empty_dirs=""
	cleanup_non_empty_dirs=""
	for n in $(seq 1 $N_STEP_SIZE $N_MAX)
	do
		dir="$DIRECTORY/$benchmark_type/$SEED/$n"
		if [ $GEOMETRIES -gt 0 ]
		then
			dir="${dir}_geom"
		fi
		mkdir -p "$dir"
		if [ "$(ls -A "$dir")" ]
		then
			non_empty_dirs="$non_empty_dirs$dir "
			cleanup_non_empty_dirs="$cleanup_non_empty_dirs$dir/* "
		fi
	done
	if [ ! -z "$non_empty_dirs" ]
	then
		if [ $CLEAN -gt 0 ]
		then
			rm $cleanup_non_empty_dirs
			if [ $INFO -gt 0 ]
			then
				echo "$0 - $(date) - INFO: Cleaned up the following directories"
				echo "$non_empty_dirs"
			fi
		else
			echo "$0 - $(date) - FATAL: The following directories are non empty:"
			echo "$non_empty_dirs"
			exit 1
		fi
	fi
}

function main() {
	initialize $@
	second_run_required=$GEOMETRIES
	GEOMETRIES=0
	generate_benchmarks
	if [ $second_run_required -gt 0 ]
	then
		if [ $INFO -gt 0 ]
		then
				echo "$0 - $(date) - INFO: Starting second benchmark generation run to create geometries"
		fi
		GEOMETRIES=1
		generate_benchmarks
	fi
}

main $@
