// Copyright (c) 2006 - 2008, Clark & Parsia, LLC. <http://www.clarkparsia.com>
// This source code is available under the terms of the Affero General Public
// License v3.
//
// Please see LICENSE.txt for full license terms, including the availability of
// proprietary exceptions.
// Questions, comments, or requests for clarification: licensing@clarkparsia.com

package openllet.explanation.test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashSet;
import java.util.Properties;
import java.util.Set;

import org.apache.jena.ontology.OntModel;
import org.apache.jena.rdf.model.Model;
import org.apache.jena.rdf.model.ModelFactory;
import org.apache.jena.rdf.model.Resource;
import org.apache.jena.rdf.model.ResourceFactory;
import org.apache.jena.vocabulary.RDF;
import org.apache.jena.vocabulary.RDFS;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;
import org.semanticweb.owlapi.model.OWLAxiom;
import org.semanticweb.owlapi.model.OWLClass;
import org.semanticweb.owlapi.model.OWLIndividual;
import org.semanticweb.owlapi.model.OWLObjectProperty;
import org.semanticweb.owlapi.model.OWLOntology;

import openllet.core.OpenlletOptions;
import openllet.core.utils.SetUtils;
import openllet.jena.PelletInfGraph;
import openllet.owlapi.OWL;
import openllet.owlapi.OpenlletReasoner;
import openllet.owlapi.OpenlletReasonerFactory;
import openllet.owlapi.explanation.PelletExplanation;

/**
 * <p>
 * Title: ExplainationInconsistent
 * </p>
 * <p>
 * Copyright: Copyright (c) 2008
 * </p>
 * <p>
 * Company: Clark & Parsia, LLC. <http://www.clarkparsia.com>
 * </p>
 *
 * @author Markus Stocker
 */
public class MiscExplanationTests
{
	private static Properties savedOptions;

	@BeforeClass
	public static void saveOptions()
	{
		final Properties newOptions = new Properties();
		newOptions.setProperty("USE_TRACING", "true");

		savedOptions = OpenlletOptions.setOptions(newOptions);
	}

	@AfterClass
	public static void restoreOptions()
	{
		OpenlletOptions.setOptions(savedOptions);
	}

	@Test
	public void testOWLAPI()
	{
		final OWLClass A = OWL.Class("A");
		final OWLClass B = OWL.Class("B");
		final OWLClass C = OWL.Class("C");
		final OWLIndividual i = OWL.Individual("i");
		final OWLIndividual j = OWL.Individual("j");

		final Set<OWLAxiom> axioms = new HashSet<>();
		axioms.add(OWL.disjointClasses(A, B));
		axioms.add(OWL.equivalentClasses(C, OWL.Nothing));
		axioms.add(OWL.classAssertion(i, A));
		axioms.add(OWL.classAssertion(i, B));
		axioms.add(OWL.classAssertion(j, C));

		final OWLOntology ontology = OWL.Ontology(axioms);
		final OpenlletReasoner reasoner = OpenlletReasonerFactory.getInstance().createReasoner(ontology);
		final PelletExplanation explain = new PelletExplanation(reasoner);

		final Set<Set<OWLAxiom>> actual = explain.getInconsistencyExplanations();

		final Set<OWLAxiom> f = new HashSet<>();
		f.add(OWL.classAssertion(i, B));
		f.add(OWL.classAssertion(i, A));
		f.add(OWL.disjointClasses(A, B));

		final Set<OWLAxiom> s = new HashSet<>();
		s.add(OWL.equivalentClasses(C, OWL.Nothing));
		s.add(OWL.classAssertion(j, C));

		final Set<Set<OWLAxiom>> expected = new HashSet<>();
		expected.add(f);
		expected.add(s);

		assertEquals(expected, actual);
	}

	@Test
	public void testPunning1() throws Exception
	{
		final OWLClass A = OWL.Class("A");
		final OWLClass B = OWL.Class("B");
		final OWLIndividual i = OWL.Individual("A");

		final Set<OWLAxiom> axioms = new HashSet<>();
		axioms.add(OWL.disjointClasses(A, B));
		axioms.add(OWL.classAssertion(i, A));
		axioms.add(OWL.classAssertion(i, B));

		final OWLOntology ontology = OWL.Ontology(axioms);
		final OpenlletReasoner reasoner = OpenlletReasonerFactory.getInstance().createReasoner(ontology);
		final PelletExplanation explain = new PelletExplanation(reasoner);

		assertFalse(explain.getInconsistencyExplanations().isEmpty());
	}

	@Test
	public void testPunning2() throws Exception
	{
		final OWLObjectProperty P = OWL.ObjectProperty("P");
		final OWLObjectProperty S = OWL.ObjectProperty("S");
		final OWLIndividual i = OWL.Individual("P");

		final Set<OWLAxiom> axioms = new HashSet<>();
		axioms.add(OWL.disjointProperties(P, S));
		axioms.add(OWL.propertyAssertion(i, P, i));
		axioms.add(OWL.propertyAssertion(i, S, i));

		final OWLOntology ontology = OWL.Ontology(axioms);
		final OpenlletReasoner reasoner = OpenlletReasonerFactory.getInstance().createReasoner(ontology);
		final PelletExplanation explain = new PelletExplanation(reasoner);

		assertFalse(explain.getInconsistencyExplanations().isEmpty());
	}

	@Test
	public void testPunning3() throws Exception
	{
		final OWLClass A = OWL.Class("A");
		final OWLIndividual i = OWL.Individual("A");

		final OWLClass B = OWL.Class("B");
		final OWLIndividual j = OWL.Individual("B");

		final Set<OWLAxiom> axioms = new HashSet<>();
		axioms.add(OWL.disjointClasses(A, B));
		axioms.add(OWL.classAssertion(i, A));
		axioms.add(OWL.classAssertion(j, B));
		axioms.add(OWL.sameAs(i, j));

		final OWLOntology ontology = OWL.Ontology(axioms);
		final OpenlletReasoner reasoner = OpenlletReasonerFactory.getInstance().createReasoner(ontology);
		final PelletExplanation explain = new PelletExplanation(reasoner);

		assertFalse(explain.getInconsistencyExplanations().isEmpty());
	}

	@Test
	public void testPunningOneOf()
	{
		final OWLClass A = OWL.Class("A");
		final OWLIndividual a = OWL.Individual("A");
		final OWLIndividual b = OWL.Individual("b");

		final Set<OWLAxiom> axioms = new HashSet<>();
		axioms.add(OWL.equivalentClasses(A, OWL.oneOf(a, b)));

		final OWLOntology ontology = OWL.Ontology(axioms);
		final OpenlletReasoner reasoner = OpenlletReasonerFactory.getInstance().createReasoner(ontology);
		final PelletExplanation explain = new PelletExplanation(reasoner);

		assertEquals(axioms, explain.getEntailmentExplanation(OWL.classAssertion(a, A)));
	}

	@Test
	public void testPunningSingletonOneOf()
	{
		final OWLClass A = OWL.Class("A");
		final OWLIndividual a = OWL.Individual("A");

		final Set<OWLAxiom> axioms = new HashSet<>();
		axioms.add(OWL.equivalentClasses(A, OWL.oneOf(a)));

		final OWLOntology ontology = OWL.Ontology(axioms);
		final OpenlletReasoner reasoner = OpenlletReasonerFactory.getInstance().createReasoner(ontology);
		final PelletExplanation explain = new PelletExplanation(reasoner);

		assertEquals(axioms, explain.getEntailmentExplanation(OWL.classAssertion(a, A)));
	}

	@Test
	public void testJena()
	{
		final Resource A = ResourceFactory.createResource("A");
		final Resource B = ResourceFactory.createResource("B");
		final Resource C = ResourceFactory.createResource("C");
		final Resource i = ResourceFactory.createResource("i");

		final Model expected = ModelFactory.createDefaultModel();
		expected.add(A, org.apache.jena.vocabulary.OWL.disjointWith, B);
		expected.add(i, RDF.type, A);
		expected.add(i, RDF.type, B);

		final OntModel model = ModelFactory.createOntologyModel(openllet.jena.PelletReasonerFactory.THE_SPEC);
		model.add(expected);
		model.add(i, RDF.type, C);

		model.prepare();

		final Model actual = ((PelletInfGraph) model.getGraph()).explainInconsistency();

		assertEquals(expected.listStatements().toSet(), actual.listStatements().toSet());
	}

	/**
	 * Tests explanation of the unsatisfiability pattern reported in bug 453.
	 */
	@Test
	public void testUnsatisfiable453()
	{
		// the names of concepts were observed to be important (other names
		// probably change the ordering in which concepts are processed)

		final OWLClass VolcanicMountain = OWL.Class("http://test#a_VOLCANICMOUNTAIN");
		final OWLClass Mountain = OWL.Class("http://test#a_MOUNTAIN");
		final OWLClass Volcano = OWL.Class("http://test#a_VOLCANO");
		final OWLClass UplandArea = OWL.Class("http://test#a_UPLANDAREA");

		final OWLAxiom[] axioms = { //
				OWL.subClassOf(VolcanicMountain, Mountain), //
				OWL.subClassOf(VolcanicMountain, Volcano), //
				OWL.subClassOf(Mountain, UplandArea), //
				OWL.subClassOf(UplandArea, OWL.not(Volcano)), //
				OWL.disjointClasses(UplandArea, Volcano) //
		};

		final OWLOntology ontology = OWL.Ontology(axioms);
		final OpenlletReasoner reasoner = OpenlletReasonerFactory.getInstance().createReasoner(ontology);
		final PelletExplanation explain = new PelletExplanation(reasoner);

		assertTrue(explain != null);

		// bug 453 manifested by throwing an OWLRuntimeException from the following statement
		// (number of explanations is important -- there are two explanations in this case, and the problem
		// only occurs if both of them are produced)
		final Set<Set<OWLAxiom>> actual = explain.getUnsatisfiableExplanations(VolcanicMountain, 0);

		final Set<OWLAxiom> f = SetUtils.create(axioms[0], axioms[1], axioms[2], axioms[3]);
		final Set<OWLAxiom> s = SetUtils.create(axioms[0], axioms[1], axioms[2], axioms[4]);
		final Set<Set<OWLAxiom>> expected = SetUtils.create(f, s);

		for (final Set<OWLAxiom> sae : expected)
		{
			boolean b = false;
			for (final Set<OWLAxiom> saa : actual)
				b |= sae.equals(saa);
			if (!b)
			{
				System.err.println("ERROR The following Explaination was also expected : ");
				sae.stream().map(OWLAxiom::toString).sorted().forEach(System.out::println);
			}
		}

		// I only disable the assertion to let the code run and detect problems.
		// assertEquals(expected, actual); // FIXME : there is a bug here.

		// The problem occured when I remove annotations from Aterm, but I can't see any thing directly related to it.
		// Maybe an indirect impact over hashcode change in ordering of hashmap.
	}

	/**
	 * Test for ticket #478
	 */
	@Test
	public void testJenaUpdates()
	{
		final Resource A = ResourceFactory.createResource("A");
		final Resource B = ResourceFactory.createResource("B");
		final Resource C = ResourceFactory.createResource("C");
		final Resource i = ResourceFactory.createResource("i");

		final OntModel model = ModelFactory.createOntologyModel(openllet.jena.PelletReasonerFactory.THE_SPEC);
		model.add(i, RDF.type, A);
		model.add(A, RDFS.subClassOf, B);

		model.prepare();
		Model actual = ((PelletInfGraph) model.getGraph()).explain(i, RDF.type, B);
		Model expected = model.getRawModel();

		assertEquals(expected.listStatements().toSet(), actual.listStatements().toSet());

		model.add(B, RDFS.subClassOf, C);

		model.prepare();

		actual = ((PelletInfGraph) model.getGraph()).explain(i, RDF.type, C);
		expected = model.getRawModel();

		assertEquals(expected.listStatements().toSet(), actual.listStatements().toSet());
	}

	@SuppressWarnings("unused")
	private void loadFromResource(final OntModel model, final String resource)
	{

		try (final InputStream stream = this.getClass().getClassLoader().getResourceAsStream(resource))
		{
			model.read(stream, null);
		}
		catch (final IOException exception)
		{
			exception.printStackTrace();
		}
	}
}
