// Copyright (c) 2006 - 2008, Clark & Parsia, LLC. <http://www.clarkparsia.com>
// This source code is available under the terms of the Affero General Public
// License v3.
//
// Please see LICENSE.txt for full license terms, including the availability of
// proprietary exceptions.
// Questions, comments, or requests for clarification: licensing@clarkparsia.com

package openllet.test.query;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import org.apache.jena.ontology.OntDocumentManager;
import org.apache.jena.ontology.OntModel;
import org.apache.jena.query.Dataset;
import org.apache.jena.query.DatasetFactory;
import org.apache.jena.query.QueryExecution;
import org.apache.jena.rdf.model.ModelFactory;
import org.apache.jena.util.FileManager;
import org.apache.jena.util.FileUtils;

import openllet.core.utils.URIUtils;
import openllet.jena.PelletInfGraph;
import openllet.jena.PelletReasonerFactory;
import openllet.jena.test.ResourceImportLoader;
import openllet.query.sparqldl.jena.SparqlDLExecutionFactory;
import openllet.query.sparqldl.jena.SparqlDLExecutionFactory.QueryEngineType;

/**
 * <p>
 * Title: Engine for processing DAWG test manifests
 * </p>
 * <p>
 * Copyright: Copyright (c) 2007
 * </p>
 * <p>
 * Company: Clark & Parsia, LLC. <http://www.clarkparsia.com>
 * </p>
 *
 * @author Petr Kremen
 */
public class PelletSparqlDawgTester extends ARQSparqlDawgTester
{
	public static boolean CLASSIFY_KB_IN_ADVANCE = false;

	private final List<String> _avoidList = Arrays.asList("open-eq-01", "open-eq-02", "open-eq-05", "open-eq-07", "open-eq-08", "open-eq-09", "open-eq-10", "open-eq-11", "open-eq-12", "term-3", "construct-1", "construct-3", "construct-4", "construct-5", "dawg-sort-builtin", "dawg-sort-function", "list-1", "list-2", "list-3", "list-4", "distinct-2", "no-distinct-2", "join-combo-1", "join-combo-2", "dawg-optional-filter-005-not-simplified", "date-2", "date-3", "unplus-1", "open-eq-03", "var-1", "var-2", "open-eq-04", "nested-opt-1", "nested-opt-2", "opt-filter-1", "opt-filter-2", "filter-scope-1", "dawg-optional-complex-2", "dawg-optional-filter-001", "dawg-optional-filter-002", "dawg-optional-filter-003", "dawg-optional-filter-005-simplified", "dawg-graph-01", "dawg-graph-03", "dawg-graph-05", "dawg-graph-06", "dawg-graph-07", "dawg-graph-08", "dawg-graph-11", "dawg-dataset-01", "dawg-dataset-03", "dawg-dataset-05", "dawg-dataset-06", "dawg-dataset-07", "dawg-dataset-08", "dawg-dataset-11", "dawg-dataset-12", "dawg-dataset-12b", "no-distinct-1", "distinct-1", "dawg-sort-4", "dawg-sort-5", "dawg-sort-7", "dawg-sort-9", "dawg-sort-10", "limit-1", "limit-2", "limit-4", "offset-1", "offset-2", "offset-4", "slice-1", "slice-2", "slice-4", "slice-5");

	protected QueryEngineType _queryEngineType;

	protected boolean _handleVariableSPO = true;

	public PelletSparqlDawgTester(final QueryEngineType queryEngineType, final boolean handleVariableSPO)
	{
		_queryEngineType = queryEngineType;
		_handleVariableSPO = handleVariableSPO;
	}

	@Override
	protected Dataset createDataset()
	{
		final boolean useQueryGraphs = !_query.getGraphURIs().isEmpty() || !_query.getNamedGraphURIs().isEmpty();

		final Collection<String> graphURIs = useQueryGraphs ? _query.getGraphURIs() : _graphURIs;
		// this handler will intercept all import resolution failures and will
		// try to load imports from resources (helps run these tests with maven)
		OntDocumentManager.getInstance().setReadFailureHandler(new ResourceImportLoader());

		OntModel model = ModelFactory.createOntologyModel(PelletReasonerFactory.THE_SPEC);

		for (final String dataURI : graphURIs)
			model.read(dataURI, FileUtils.guessLang(dataURI));

		model.prepare();

		if (PelletSparqlDawgTester.CLASSIFY_KB_IN_ADVANCE)
			((PelletInfGraph) model.getGraph()).getKB().classify();

		final Dataset dataset = DatasetFactory.create(model);

		final Collection<String> namedGraphURIs = useQueryGraphs ? _query.getNamedGraphURIs() : _namedGraphURIs;

		for (final String graphURI : namedGraphURIs)
		{
			model = ModelFactory.createOntologyModel(PelletReasonerFactory.THE_SPEC);
			FileManager.get().readModel(model, graphURI);
			dataset.addNamedModel(graphURI, model);
		}

		return dataset;
	}

	@Override
	protected QueryExecution createQueryExecution()
	{
		return SparqlDLExecutionFactory.create(_query, createDataset(), null, _queryEngineType, _handleVariableSPO);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isApplicable(final String uri)
	{
		return !_avoidList.contains(URIUtils.getLocalName(uri));
	}

	@Override
	public String getName()
	{
		return _queryEngineType.toString();
	}
}
